import { NextRequest, NextResponse } from 'next/server'
import { prisma } from '@/lib/db'
import { auth } from '@/auth'

// GET - Tek bir blog yazısını getir
export async function GET(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await auth()
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { error: 'Yetkisiz erişim' },
        { status: 401 }
      )
    }

    const post = await prisma.blogPost.findUnique({
      where: { id: params.id },
    })

    if (!post) {
      return NextResponse.json(
        { error: 'Blog yazısı bulunamadı' },
        { status: 404 }
      )
    }

    return NextResponse.json(post)
  } catch (error) {
    console.error('Blog yazısı yüklenirken hata:', error)
    return NextResponse.json(
      { error: 'Blog yazısı yüklenemedi' },
      { status: 500 }
    )
  }
}

// PUT - Blog yazısını güncelle
export async function PUT(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await auth()
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { error: 'Yetkisiz erişim' },
        { status: 401 }
      )
    }

    const body = await request.json()
    const { title, slug, excerpt, content, category, coverImage, author, published } = body

    // Slug değiştirilmişse ve başka bir yazıda kullanılmışsa hata ver
    if (slug) {
      const existingPost = await prisma.blogPost.findUnique({
        where: { slug },
      })

      if (existingPost && existingPost.id !== params.id) {
        return NextResponse.json(
          { error: 'Bu slug zaten kullanılıyor' },
          { status: 400 }
        )
      }
    }

    const updateData: any = {
      ...(title && { title }),
      ...(slug && { slug }),
      ...(excerpt && { excerpt }),
      ...(content && { content }),
      ...(category && { category }),
      ...(author && { author }),
    }

    if (coverImage !== undefined) {
      updateData.coverImage = coverImage || null
    }

    if (published !== undefined) {
      updateData.published = published
      if (published) {
        const currentPost = await prisma.blogPost.findUnique({
          where: { id: params.id },
        })
        if (!currentPost?.publishedAt) {
          updateData.publishedAt = new Date()
        }
      }
    }

    const post = await prisma.blogPost.update({
      where: { id: params.id },
      data: updateData,
    })

    return NextResponse.json(post)
  } catch (error) {
    console.error('Blog yazısı güncellenirken hata:', error)
    return NextResponse.json(
      { error: 'Blog yazısı güncellenemedi' },
      { status: 500 }
    )
  }
}

// DELETE - Blog yazısını sil
export async function DELETE(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await auth()
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { error: 'Yetkisiz erişim' },
        { status: 401 }
      )
    }

    await prisma.blogPost.delete({
      where: { id: params.id },
    })

    return NextResponse.json({ success: true })
  } catch (error) {
    console.error('Blog yazısı silinirken hata:', error)
    return NextResponse.json(
      { error: 'Blog yazısı silinemedi' },
      { status: 500 }
    )
  }
}



